using System;
using System.Collections;
using System.Data;
using System.Data.SqlClient;
using System.Reflection;

using Sprocs = gov.va.med.vbecs.Common.VbecsStoredProcs;
using DivTable = gov.va.med.vbecs.Common.VbecsTables.VamcDivision;

using gov.va.med.vbecs.Common;

namespace gov.va.med.vbecs.DAL
{

	#region Header

	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///	<Developer>Cameron Taylor</Developer>
	///</Developers>
	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate>11/4/2002</CreationDate>
	///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	///<summary>
	///	Data access component implementing database read/write functionality
	///	for division-related data.
	///</summary>

	#endregion

	public class Division
	{
		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/30/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="770"> 
		///		<ExpectedInput>Valid division code</ExpectedInput>
		///		<ExpectedOutput>Data row containing division details</ExpectedOutput>
		///	</Case>
		///<Case type="1" testid ="772"> 
		///		<ExpectedInput>Invalid division code</ExpectedInput>
		///		<ExpectedOutput>DataAccessLayerException raised</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Retrieves a division identified by supplied division code from database.
		/// </summary>
		/// <param name="divisionCode">Division code.</param>
		/// <returns>DataRow containing division details</returns>
		public static DataRow GetDivision( string divisionCode )
		{
			SqlParameter prm =	new SqlParameter(Sprocs.GetDivision.divisioncode, System.Data.SqlDbType.Char );
			prm.Value = divisionCode;

			return StoredProcedure.GetSingleDataRow(Sprocs.GetDivision.StoredProcName, prm );
		}
		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/27/2007</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8502"> 
		///		<ExpectedInput>Valid division code</ExpectedInput>
		///		<ExpectedOutput>Datatable containin</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="8503"> 
		///		<ExpectedInput>null parameter</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets all institutions associated with a given division CR 2316, DR 2723 
		/// </summary>
		/// <param name="divisionCode"></param>
		/// <returns></returns>
		public static DataTable GetAssociatedInstitutionsByDivision( string divisionCode )
		{
			if( divisionCode == null )
				throw( new ArgumentNullException( "divisionCode" ) ); 

			SqlParameter prm =	new SqlParameter(Sprocs.GetAssociatedInstitutionByDivision.divisioncode, System.Data.SqlDbType.Char );
			prm.Value = divisionCode;

			return StoredProcedure.GetDataTable(Sprocs.GetAssociatedInstitutionByDivision.StoredProcName, prm );
		}
		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/27/2007</CreationDate>
		///<Case type="0" testid ="8506"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>a datatable of all associated institutions, if none returns null</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="8507"> 
		///		<ExpectedInput>n/a</ExpectedInput>
		///		<ExpectedOutput>a datatable of all associated institutions, if none returns null</ExpectedOutput>
		///	</Case>
		///
		///
		/// <summary>
		/// Gets all institutions associated 
		/// </summary>
		/// <returns></returns>
		public static DataTable GetVbecsAssociatedInstitution( )
		{
			System.Data.DataSet ds = StoredProcedure.GetData(Sprocs.GetVbecsAssociatedInstitution.StoredProcName);
			if(ds.Tables.Count == 1)
			{
				return ds.Tables[0];
			}
			else
			{
				return null;
			}
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/30/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="771"> 
		///		<ExpectedInput>False</ExpectedInput>
		///		<ExpectedOutput>DataTable containing data for all divisions.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="7182"> 
		///		<ExpectedInput>True</ExpectedInput>
		///		<ExpectedOutput>DataTable containing data for active divisions only.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4823"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Retrieves all divisions in VBECS DB.
		/// </summary>
		/// <returns>DataTable containing list of all divisions.</returns>
		public static DataTable GetDivisions( bool activeOnly )
		{
			SqlParameter activeOnlyParam = new SqlParameter( Sprocs.GetVbecsDivisions.activeonly, SqlDbType.Bit );
			activeOnlyParam.Value = activeOnly;

			return StoredProcedure.GetDataTable( Sprocs.GetVbecsDivisions.StoredProcName, activeOnlyParam );				
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/30/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="769"> 
		///		<ExpectedInput>division code, electronic crossmatch indicator, iccbba registration number, row version</ExpectedInput>
		///		<ExpectedOutput>Success</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="1863"> 
		///		<ExpectedInput>Bad RowVersion</ExpectedInput>
		///		<ExpectedOutput>RowVersionException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Updates division details in database and returns success indicator
		/// </summary>
		/// <param name="divisionCode">Division code</param>
		/// <param name="electronicCrossmatchIndicator">Electronic crossmatch indicator</param>
		/// <param name="iccbbaRegistrationNumber">ICCBBA registration number</param>
		/// <param name="antibodyScreenDisplayCode">Antibody screen display code</param>
		/// <param name="automatedInstrumentIndicator">Automated instrument indicator</param>
		/// <param name="medicalDirectorName">Medical Director Name</param>
		/// <param name="iconAlertsIndicator">Icon alerts indicator</param>
		/// <param name="iconOrderAlertType">Icon order alert type</param>
		/// <param name="iconOrdersNeededHours">Icon orders needed hours</param>
		/// <param name="iconRefreshRate">Icon refresh rate</param>
		/// <param name="printAlertsIndicator">Print alerts indicator</param>
		/// <param name="printerName">Printer name</param>
		/// <param name="printOrderAlertType">Print order alert type</param>
		/// <param name="printOrderNeededHours">Print order needed hours</param>
		/// <param name="loginMessage">Login message</param>
		/// <param name="maximumSpecimenExpirationDays">
		///		Maximum for a number of 
		///		specimen expiration days
		///	</param>
		/// <param name="rowVersion">Rowversion</param>		
		/// <param name="lastUpdateFunctionId">UC or calling method</param>
		/// <param name="reactivityPhaseCode"></param>
		/// <returns>True on success. Otherwise - false.</returns>
		public static bool UpdateDivision(
			string divisionCode, 
			bool electronicCrossmatchIndicator, 
			string iccbbaRegistrationNumber,
			int antibodyScreenDisplayCode,
			bool automatedInstrumentIndicator,
			string medicalDirectorName,
			// Division Alerts
			bool iconAlertsIndicator,
			Common.OrderAlertTypeCode iconOrderAlertType,
			int iconOrdersNeededHours,
			int iconRefreshRate,
			bool printAlertsIndicator,
			string printerName,
			Common.OrderAlertTypeCode printOrderAlertType,
			int printOrderNeededHours,
			string loginMessage,
			int maximumSpecimenExpirationDays,
			byte[] rowVersion,
			Common.UpdateFunction lastUpdateFunctionId,
			Common.ReactivityPhase reactivityPhaseCode)
		{

			SqlParameter[] prms = 
			{			
				new SqlParameter (Sprocs.UpdateDivision.divisioncode, System.Data.SqlDbType.Char),
				new SqlParameter (Sprocs.UpdateDivision.electroniccrossmatchindicator, System.Data.SqlDbType.Bit),
				new SqlParameter (Sprocs.UpdateDivision.iccbbaregistrationnumber, System.Data.SqlDbType.Char),
				new SqlParameter (Sprocs.UpdateDivision.antibodyscreendisplaycode, System.Data.SqlDbType.Int),
				new SqlParameter (Sprocs.UpdateDivision.autoinstrumentindicator, System.Data.SqlDbType.Bit),
				new SqlParameter (Sprocs.UpdateDivision.medicaldirectorname, System.Data.SqlDbType.VarChar),
				// Division Alerts
				new SqlParameter (Sprocs.UpdateDivision.iconalertsindicator, System.Data.SqlDbType.Bit),
				new SqlParameter (Sprocs.UpdateDivision.iconorderalerttypecode, System.Data.SqlDbType.Char),
				new SqlParameter (Sprocs.UpdateDivision.iconordersneededhours, System.Data.SqlDbType.Int),
				new SqlParameter (Sprocs.UpdateDivision.iconrefreshrate, System.Data.SqlDbType.Int),
				new SqlParameter (Sprocs.UpdateDivision.printeralertsindicator, System.Data.SqlDbType.Bit),
				new SqlParameter (Sprocs.UpdateDivision.printeralertsprintername , System.Data.SqlDbType.VarChar),
				new SqlParameter (Sprocs.UpdateDivision.printerorderalerttypecode , System.Data.SqlDbType.Char),
				new SqlParameter (Sprocs.UpdateDivision.printerordersneededhours , System.Data.SqlDbType.Int),
				new SqlParameter (Sprocs.UpdateDivision.loginmessagetext , System.Data.SqlDbType.VarChar),

				new SqlParameter (Sprocs.UpdateDivision.maximumspecimenexpirationdays , System.Data.SqlDbType.Int),
				new SqlParameter (Sprocs.UpdateDivision.lastupdateuser, System.Data.SqlDbType.VarChar),
				new SqlParameter (Sprocs.UpdateDivision.rowversion, System.Data.SqlDbType.Binary),
				new SqlParameter (Sprocs.UpdateDivision.lastupdatefunctionid, System.Data.SqlDbType.Int),
				//CR 2717
				new SqlParameter (Sprocs.UpdateDivision.reactivityphasecode, System.Data.SqlDbType.Char)
			};

			prms[0].Value = divisionCode;
			prms[1].Value = electronicCrossmatchIndicator;
			prms[2].Value = iccbbaRegistrationNumber;
			prms[3].Value = antibodyScreenDisplayCode;
			prms[4].Value = automatedInstrumentIndicator;
			prms[5].Value = medicalDirectorName;
				
			prms[6].Value = iconAlertsIndicator;
			prms[7].Value = Common.Utility.GetOrderAlertTypeTypeFromEnum(iconOrderAlertType);
			prms[8].Value = iconOrdersNeededHours;
			prms[9].Value = iconRefreshRate;
			prms[10].Value = printAlertsIndicator;
			prms[11].Value = printerName;
			prms[12].Value = Common.Utility.GetOrderAlertTypeTypeFromEnum(printOrderAlertType);
			prms[13].Value = printOrderNeededHours;
			prms[14].Value = loginMessage;

			prms[15].Value = maximumSpecimenExpirationDays;
			prms[16].Value = Common.LogonUser.LogonUserName;
			prms[17].Value = rowVersion;
			prms[18].Value = (int)lastUpdateFunctionId;
			//CR 2717
			prms[19].Value = Common.Utility.GetReactivityCharFromCode(reactivityPhaseCode);

			return ( new StoredProcedure().TransactionalGetValue( Sprocs.UpdateDivision.StoredProcName, prms ) == 0 );
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/30/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1173"> 
		///		<ExpectedInput>division</ExpectedInput>
		///		<ExpectedOutput>Success</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="1175"> 
		///		<ExpectedInput>null division</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Public static method returning a list of all medical director names entered for given division
		/// </summary>
		/// <param name="divisionCode">Division Code</param>
		/// <returns>Medical Director names</returns>
		public static DataTable GetMedicalDirectorNames( string divisionCode )
		{
			if( divisionCode == null )
				throw( new ArgumentNullException( "divisionCode" ) ); 

			SqlParameter _prm =	new SqlParameter( Sprocs.GetMedicalDirectors.DivisionCode, System.Data.SqlDbType.Char );

			_prm.Value = divisionCode;

			return StoredProcedure.GetDataTable( Sprocs.GetMedicalDirectors.StoredProcName, _prm );
		}

		// TODO: add unit tests
		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/22/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8246"> 
		///		<ExpectedInput>division</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="8247"> 
		///		<ExpectedInput>null</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns a boolean telling if units exist in the division which are not in a final status
		/// BR_9.09
		/// </summary>
		/// <returns></returns>
		public static bool IsBloodUnitNotInFinalStatus( string divisionCode )
		{
			if( divisionCode == null )
				throw( new ArgumentNullException( "divisionCode" ) );

			SqlParameter[] prms =
			{
				new SqlParameter(Sprocs.IsBloodUnitNotInFinalStatus.divisioncode, System.Data.SqlDbType.VarChar)
			};

			prms[0].Value = divisionCode;

			return new StoredProcedure().GetBoolReturnValue( Sprocs.IsBloodUnitNotInFinalStatus.StoredProcName, prms );
		}


		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/13/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8248"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="8249"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// PersistDivisionBaseData
		/// </summary>
		/// <param name="isNewDivisionBeingAdded"></param>
		/// <param name="divBaseData"></param>
		/// <param name="isNewLocalSupplier"></param>
		/// <param name="addressData"></param>
		/// <param name="facilityData"></param>
		/// <param name="isLocalSupplierInsertingNewAddress"></param>
		/// <param name="vbecsUsersToSaveData"></param>
		/// <param name="vbecsUsersToSaveDivisionRolesData"></param>
		/// <param name="associatedInstitutions"></param>
		/// <param name="lastUpdateFunction"></param>
		public static void PersistDivisionBaseData( bool isNewDivisionBeingAdded, DataTable divBaseData, DataTable facilityData, DataTable addressData, bool isNewLocalSupplier, bool isLocalSupplierInsertingNewAddress, DataTable vbecsUsersToSaveData, IList vbecsUsersToSaveDivisionRolesData, DataTable associatedInstitutions, UpdateFunction lastUpdateFunction )
		{
			if( divBaseData == null )
				throw( new ArgumentNullException( "divBaseData" ) );

			if( divBaseData.Rows.Count != 1 )
				throw( new ArgumentException( StrRes.SysErrMsg.MUC02.DivisionBaseUpdateInvalidInputRowCount( divBaseData.Rows.Count ).ResString ) );

			ArrayList	dataTables = new ArrayList( 5 ),
						storedProcs = new ArrayList( 5 );

			storedProcs.Add( isNewDivisionBeingAdded ? VbecsStoredProcs.InsertVamcDivisionBase.StoredProcName : VbecsStoredProcs.UpdateVamcDivisionBase.StoredProcName );
			dataTables.Add( Common.Utility.AppendLastUpdateInformation( divBaseData, lastUpdateFunction ) );

			if( isNewDivisionBeingAdded )
				AddDivisionDayOneTemplatesInsertActions( storedProcs, dataTables, (string)divBaseData.Rows[0][ VbecsTables.VamcDivision.DivisionCode ], lastUpdateFunction );

			if( facilityData != null )
			{
				//Append the DataTable and Sproc names to the ArrayList
				dataTables.Add(addressData);
				dataTables.Add(facilityData);

				//Address goes first since LocalSupplier is dependent on it
				if (isLocalSupplierInsertingNewAddress)
					storedProcs.Add(Sprocs.InsertAddress.StoredProcName);
				else
					storedProcs.Add(Sprocs.UpdateAddress.StoredProcName);

				if (isNewLocalSupplier )
					storedProcs.Add(Sprocs.InsertLocalSupplier.StoredProcName);
				else
					storedProcs.Add(Sprocs.UpdateLocalSupplier.StoredProcName);

			}
			
			if(associatedInstitutions != null)
			{
				System.Data.DataTable dtNewAssociations = associatedInstitutions.GetChanges(System.Data.DataRowState.Added);
				if(dtNewAssociations != null)
				{
					storedProcs.Add(Sprocs.InsertAssociatedInstitution.StoredProcName);
					dataTables.Add(dtNewAssociations);
				}
				System.Data.DataTable dtChangedAssociations = associatedInstitutions.GetChanges(System.Data.DataRowState.Modified);
				if(dtChangedAssociations != null)
				{
					storedProcs.Add(Sprocs.UpdateAssociatedInstitution.StoredProcName);
					dataTables.Add(dtChangedAssociations);
				}
					
			}

			VbecsUser.AddVbecsUsersListSaveActions( storedProcs, dataTables, vbecsUsersToSaveData, vbecsUsersToSaveDivisionRolesData, lastUpdateFunction );

			new StoredProcedure().TransactionalGetValue( storedProcs, dataTables );
		}

		/// <summary>
		/// Adds necessary data save actions for inserting day one division templates
		/// by adding data and sproc names to the given lists so those 
		/// lists may be passed further for updating base division data in the DB.
		/// </summary>
		/// <param name="storedProcs">List of stored procedures to add to.</param>
		/// <param name="dataTables">List of data tables to add to.</param>
		/// <param name="divisionCode">Division code. </param>
		/// <param name="lastUpdateFunction">Last update function ID.</param>
		private static void AddDivisionDayOneTemplatesInsertActions( IList storedProcs, IList dataTables, string divisionCode, UpdateFunction lastUpdateFunction )
		{
			if( storedProcs == null )
				throw( new ArgumentNullException( "storedProcs" ) );

			if( dataTables == null )
				throw( new ArgumentNullException( "dataTables" ) );

			if( divisionCode == null )
				throw( new ArgumentNullException( "divisionCode" ) );
			
			DataTable _dt = new DataTable();
			_dt.Columns.Add( VbecsTables.VamcDivision.DivisionCode, typeof(string) );
			_dt.Rows.Add( new object[]{ divisionCode } );

			storedProcs.Add( VbecsStoredProcs.SeedAllDayOneTemplatesForDivision.StoredProcName );
			dataTables.Add( Common.Utility.AppendLastUpdateInformation( _dt, lastUpdateFunction ) );
		}
	}
}